//
//  GlyphsToolEventProtocol.h
//  TestToolPlugin
//
//  Created by Georg Seifert on 4.7.07.
//  Copyright 2007 schriftgestaltung.de. All rights reserved.
//

#import <GlyphsCore/GlyphsToolAuxiliaryDrawing.h>

extern NSUInteger GSEventModifierFlagSpace;

@class GSLayer;

@protocol GSGlyphViewControllerProtocol;
@protocol GSGlyphEditViewControllerProtocol;

extern CGFloat ClickTolerance;

/** Protocol for the Event handling parts of the tool API.

 All plugins should conform to this protocol. Additionally it could conform the GlyphsToolDraw it it needs to draw.

 The tool can implement all methods that are defined by [NSResponder](https://developer.apple.com/library/mac/documentation/Cocoa/Reference/ApplicationKit/Classes/NSResponder_Class/)

 @see GlyphsToolDraw
*/
@protocol GlyphsToolEvent <GlyphsToolAuxiliaryDrawing>

/// The interface version.
@property (nonatomic, readonly) NSUInteger interfaceVersion;

/** The group ID defines the position of the tool in the toolbar.

- selection tools (select, text): 10
- editTools (draw): 20
- navigationTools (hand, zoom): 30
*/
@property (nonatomic, readonly) NSUInteger groupID;

/// The description/name of the plugin.
@property (nonatomic, readonly, nonnull) NSString *title;

/// The icon displayed in the toolbar.
@property (nonatomic, readonly, nonnull) NSImage *toolBarIcon;

/// The default keyboard shortcut to select the tool.
@property (nonatomic, readonly, nullable) NSString *trigger;

/** The trigger (key mask) to select the tool temporarily

 The select tool is selected with the cmd key. It returns NSCommandKeyMask.
 */
@property (nonatomic, readonly) NSEventModifierFlags tempTrigger;

/** Set to YES while the mouse is down and dragging.

The setter should initialize the drag action. (set dragStart)
*/
@property (nonatomic) BOOL dragging;

/// The position where the dragging action started.
@property (nonatomic) NSPoint dragStart;

/** Informs the receiver that the user has pressed or released a modifier key (Shift, Control, and so on).

 @param anEvent An object encapsulating information about the modifier-key event.
 */
- (void)modifierChanged:(nonnull NSEvent *)anEvent;

/** Is called when the tool is activated.

This might happen when the user selects the tool or when the document becomes active
@see willDeactivate
*/
- (void)willActivate;

/** Is called when the tool is deactivated.

 This might happen when the user deselects the tool or when the document becomes inactive
 @see willActivate
 */
- (void)willDeactivate;

/**
 A callback to let the tool know that a temp tool will be selected (by pressing a modifier key). You can return NO to prevent that.

 @param tempTool the temp tool that is about to become active

 @return if you allow it or not
 */
- (BOOL)willSelectTempTool:(nonnull id)tempTool;

/** Draws the active selection.
  The default is a colored drag rect.
 */
- (void)drawSelection;

@optional

/** Copy'n'Past Support */
- (void)copy:(nullable id)sender;

/** Copy'n'Past Support */
- (void)paste:(nullable id)sender;

/** Return a NSMenu object that should be displayed if the user presses/holds the mouse on the icon

@return A NSMenu object
*/
- (nullable NSMenu *)toolbarMenu;

- (nullable NSResponder<GlyphsToolEvent> *)eventHandler;

- (void)mouseDoubleDown:(nonnull NSEvent *)theEvent;

- (nullable NSMenu *)defaultContextMenu;

/**
 Callback to allow to add something to the context menu

 @param theEvent the current event
 @param theMenu  the context menu to add to
 */
- (void)addMenuItemsForEvent:(nonnull NSEvent *)theEvent toMenu:(nonnull NSMenu *)theMenu;

- (void)addMenuItemsForEvent:(nonnull NSEvent *)theEvent controller:(nonnull NSViewController<GSGlyphEditViewControllerProtocol> *)controller toMenu:(nonnull NSMenu *)theMenu;
/**
 if the tool has several sub groups, this is called to switch to the next tool

 @param sender the object calling the command
 */
- (void)selectNextSubTool:(nullable id)sender;

/**
 returns all layers that are currently active. That is mostly used for the 'Select All Layers' tool.

 @return an array of layers
 */
- (nonnull NSArray *)activeLayers;

/// returns the view to place in the inspector
- (nullable NSArray *)inspectorViewControllers;

/// returns the view to place on the left of the normal inspector
- (nullable NSArray *)extraInspectorViewControllers;

- (nullable NSArray *)panelInspectors;

/// return true if the current selection state might show a "Info for Selection" dialog.
- (BOOL)canShowInfoForSelection;
/** actually show the info for Selection dialog

 is only called if canShowInfoForSelection returned YES.

 @see canShowInfoForSelection
 */
- (void)showInfoForSelection;

- (int)previewOrientation;
@end

@protocol GSInspectorViewControllerDelegate <NSObject>

- (nullable NSArray *)inspectorViewControllersForLayer:(nonnull GSLayer *)layer;

/// used form the Edit View
- (nullable NSArray *)panelInspectorsForLayer:(nonnull GSLayer *)layer;

/// used form the Font View
- (nullable NSArray *)panelInspectorsForSelectedLayers:(nonnull NSArray *)layers;

@end

@protocol GSContextMenuCallbackProtocol <NSObject>
#ifndef GLYPHS_LITE
+ (void)contextMenuCallback:(nonnull NSMenu *)menu forSelectedLayers:(nonnull NSArray *)layers event:(nullable NSEvent *)event;
#endif
@end

@protocol GSSelectionChangeCallbackProtocol <NSObject>
#ifndef GLYPHS_LITE
+ (void)selectionDidChangeInLayer:(nonnull GSLayer *)layer;
#endif
@end
